/*
  * This sketch illustrates opening files by index
  * which can significantly reduce latency.
 
  * Copy all the files (20 files in total, named as waveX.wav)in the vibration folder. 
  * Then, run this sketch using the prepared SD. 
   
  * For details of the project and the circuit required, 
  * please refer to our project page at:  
  * https://userinterfaces.aalto.fi/button-design
 
  Our paper can be found at:
  * https://dx.doi.org/10.1145/3313831.3376262 
  * https://arxiv.org/abs/2001.04352
  
  The code is supposed to be run on Arduino Uno
  * https://www.amazon.com/s?k=arduino+uno
  
  Created day 2020 Jan. 15
  By Yi-Chi Liao (yi-chi.liao@aalto.fi)
  Any questions regarding installing and running the system, please contact: yi-chi.liao@aalto.fi
*/

#include <WaveHC.h>
#include <WaveUtil.h>
#include <Wire.h>

SdReader card;    // This object holds the information for the card
FatVolume vol;    // This holds the information for the partition on the card
FatReader root;   // This holds the information for the volumes root directory
FatReader file;   // This object represent the WAV file 
WaveHC wave;      // This is the only wave (audio) object, since we will only play one at a time

int LED = 13;
int x = -1;

// Number of files.
#define FILE_COUNT 21

// time to play each tone in milliseconds
#define PLAY_TIME 100

/*
 * Define macro to put error messages in flash memory
 */
#define error(msg) error_P(PSTR(msg))

//////////////////////////////////// SETUP
void setup() {
  Serial.begin(9600);

  if (!card.init()) error("card.init");

  // enable optimized read - some cards may timeout
  card.partialBlockRead(true);

  if (!vol.init(card)) error("vol.init");

  if (!root.openRoot(vol)) error("openRoot");

  PgmPrintln("Index files");
  indexFiles();

  //////////////
  pinMode (LED, OUTPUT);
  // Start the I2C Bus as a Slave on address 9
  Wire.begin(9); 
  // Attach a function to trigger when something is received.
  Wire.onReceive(receiveEvent);
}

//////////////////////////////////// LOOP
void loop() 
{ 
  //If value received is 0 blink LED for 200 ms
  Wire.onReceive(receiveEvent);
  if (x != -1) {
    //Serial.println(x);
    playNumber(x);
    x = -1;
  }
}

void(* resetFunc) (void) = 0; //declare reset function @ address 0
void receiveEvent(int bytes) {
  x = Wire.read();    // read one character from the I2C
}

/*
 * print error message and halt
 */
void error_P(const char *str) {
  PgmPrint("Error: ");
  SerialPrint_P(str);
  sdErrorCheck();
  while(1);
}
/*
 * print error message and halt if SD I/O error, great for debugging!
 */
void sdErrorCheck(void) {
  if (!card.errorCode()) return;
  PgmPrint("\r\nSD I/O error: ");
  Serial.print(card.errorCode(), HEX);
  PgmPrint(", ");
  Serial.println(card.errorData(), HEX);
  while(1);
}

// file names are of the form wavex.WAV where x is one of
// the letters from fileLetter[]
char fileLetter[] =  {'0', '1', '2', '3', '4', '5', '6', 
      '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K'}; 
      
// index of DTMF files in the root directory
uint16_t fileIndex[FILE_COUNT];
/*
 * Find files and save file index.  A file's index is is the
 * index of it's directory entry in it's directory file. 
 */
void indexFiles(void) {
  char name[10];
  
  // copy flash string to RAM
  strcpy_P(name, PSTR("wavex.WAV"));
  Serial.println(name);
  for (uint8_t i = 0; i < FILE_COUNT; i++) {
    
    // Make file name
    name[4] = fileLetter[i];
    
    // Open file by name
    if (!file.open(root, name)) error("open by name");
    
    // Save file's index (byte offset of directory entry divided by entry size)
    // Current position is just after entry so subtract one.
    fileIndex[i] = root.readPosition()/32 - 1;   
  }
  PgmPrintln("Done");
}

void playNumber(int count) {
  
  // open by index
  if (count >FILE_COUNT)
  {count = FILE_COUNT-1;}
  Serial.println(count);
  Serial.println(fileIndex[count]);

  if (!file.open(root, fileIndex[count])) {
    error("open by index");
    //resetFunc();
  }
    
    // create and play Wave
  if (!wave.create(file)) {
    error("wave.create");
    //resetFunc();
  }
  wave.play();
  delay(50);
    
  // check for play errors
  sdErrorCheck();
  
  PgmPrintln("Done");
}
